"""The configuration of the project."""
import pprint


class ConfigDict():
    _frozen = False

    def __getattr__(self, item):
        if self._frozen or item.startswith('_'):
            raise AttributeError(item)

        ret = ConfigDict()
        setattr(self, item, ret)
        return ret

    def __setattr__(self, key, value):
        if self._frozen and key not in self.__dict__:
            raise AttributeError('Unknown config: {:s}'.format(key))
        super().__setattr__(key, value)

    def __str__(self):
        return pprint.pformat(self.to_dict(), indent=1, width=100, compact=True)

    def to_dict(self):
        return {k: v.to_dict() if isinstance(v, ConfigDict) else v for k, v in self.__dict__.items() if
                not k.startswith('_')}

    def from_dict(self, d):
        self.freeze(False)
        for k, v in d.items():
            self_v = getattr(self, k)
            if isinstance(v, ConfigDict):
                self_v.from_dict(v)
            else:
                setattr(self, k, v)

    def freeze(self, frozen=True):
        self._frozen = frozen
        for v in self.__dict__.values():
            if isinstance(v, ConfigDict):
                v.freeze(frozen)

    def __eq__(self, _):
        raise NotImplementedError()

    def __ne__(self, _):
        raise NotImplementedError()


config = ConfigDict()
_C = config

# debug
_C.DEBUG.TEST_DIR = 'reid/debug/'  # offline debug images folder
_C.DEBUG.INIT_REID = True  # remove 'face encoding', 'face unknown', 'gallery'
_C.DEBUG.INIT_ATTR = True  # remove 'results.txt', 'reid unknown'

# yolov3
_C.YOLO.CFG = 'yolov3/cfg/yolov3.cfg'
_C.YOLO.DATA_CFG = 'yolov3/data/coco.data'
_C.YOLO.WEIGHTS = 'yolov3/weights/yolov3.pt'
_C.YOLO.SOURCE = 'attribute_detect/video_front'
_C.YOLO.OUTPUT = 'yolov3/output'
_C.YOLO.IMG_SIZE = 416
_C.YOLO.CONF_TH = 0.7
_C.YOLO.NMS_TH = 0.5
_C.YOLO.FOURCC = 'mp4v'
_C.YOLO.GPU_ID = 0
_C.YOLO.TRACK_TH = 50
_C.YOLO.REID_RECV = None
_C.YOLO.REID_PUT = None
_C.YOLO.FACE_RECV = None
_C.YOLO.FACE_PUT = None
_C.YOLO.BODY_RECV = None
_C.YOLO.BODY_PUT = None

# attribute detection model
_C.ATTR.PIPE = None
_C.ATTR.DATASET = 'market'
_C.ATTR.MODEL = 'resnet50'
_C.ATTR.ENCODING = 'face_detect/encoding.npy'
_C.ATTR.GALLERY_NUM = 7  # the image number of each person stored in gallery
_C.ATTR.UNKNOWN = 'reid/test_data/face/unknown'
_C.ATTR.IMG_QUEUE = None  # store thr queue create by start_capture function
_C.ATTR.IMGS = None  # store the images fetch by capture function
_C.ATTR.CAMERA_FPS = 25  # the FPS of the attribute camera
_C.ATTR.DET_FPS = 16  # the FPS of the detection action, bigger if want more precise results
_C.ATTR.HIST_NUM = 512  # the history that match face with body, bigger if network unstable
_C.ATTR.FACE_CAM = 'c5'  # the camera used to catch face info
_C.ATTR.BODY_CAM = 'c6'

# reid 3dta model
_C.REID.GPU_ID = 1  # the gpu that 3dta uses
_C.REID.WHICH_EPOCH = '120'
_C.REID.TEST_DIR = 'reid/test_data'
_C.REID.RESULT_DIR = 'reid/debug/results'
_C.REID.NAME = 'reid/ResNet50'
_C.REID.BATCHSIZE = 16  # batch size for 3dta model
_C.REID.DATASETS = ['gallery', 'query', 'face']
_C.REID.UNKNOWN = 'reid/test_data/query/unknown'
_C.REID.GALLERY = 'reid/test_data/gallery'
_C.REID.RESULT = 'reid/results.txt'
_C.REID.CAMERA_FPS = 25
_C.REID.DET_FPS = 8
_C.REID.TH = 0.50  # the threshold use for reid when statistic time
_C.REID.LIB_TH = 0.9  # the threshold use for reid when add people to library
_C.REID.IMG_QUEUE = None  # store the queue create by start_capture function
_C.REID.IMGS = None  # store the images fetch by capture function
_C.REID.ONLINE = True  #  online or offline
_C.REID.ENABLE_HIST = True  # True for enable histogram tracking strategy
_C.REID.SAVE_IMG = True  # save images in folder reid.result_dir
_C.REID.SAVE_VIDEO = False  # save videos in folder reid.result_dir
_C.REID.SAVE_VIDEO_FPS = 15  # save videos in folder reid.result_dir

# camera urls
_C.CAMERA.REID_URLS = [  # list all cameras as follow: (camera_id, camera_url)
    # ('c1', 'rtsp://admin:2284424q@172.18.232.74:70/'),  # 70
    # ('c2', 'rtsp://admin:2284424q@172.18.232.74:72/')]  # 72
    # ('c1', 'reid/debug/c1.mp4'),
    # ('c2', 'reid/debug/c2.mp4'),
    # ('c3', 'reid/debug/c3.mp4'),
    # ('c4', 'reid/debug/c4.mp4')]
    ('c1', 'rtmp://172.18.36.198:1935/hls/c1'),
    ('c2', 'rtmp://172.18.36.198:1935/hls/c2'),
    ('c3', 'rtmp://172.18.36.198:1935/hls/c2'),
    ('c4', 'rtmp://172.18.36.198:1935/hls/c4')]
_C.CAMERA.FACE_URLS = [
    # (_C.ATTR.FACE_CAM, 'rtmp://172.18.160.122:1935/live/la'),
    # (_C.ATTR.BODY_CAM, 'rtmp://172.18.160.122:1935/live/lv')]
    # (_C.ATTR.FACE_CAM, 'rtsp://admin:2284424q@172.18.232.74:71'),
    # (_C.ATTR.BODY_CAM, 'rtsp://admin:2284424q@172.18.232.74:73')]
    (_C.ATTR.FACE_CAM, 'rtmp://172.18.36.198:1935/hls/face_c1'),
    (_C.ATTR.BODY_CAM, 'rtmp://172.18.36.198:1935/hls/face_c2')]
_C.CAMERA.DEBUG = [
    (_C.ATTR.FACE_CAM, 'reid/debug/192.168.1.71.avi'),
    (_C.ATTR.BODY_CAM, 'reid/debug/192.168.1.73.avi')]

config.freeze()
