import os
import shutil
from multiprocessing import Process, Queue

import torch

from yolov3.models import Darknet, load_darknet_weights  # set ONNX_EXPORT in models.py
from yolov3.utils.utils import non_max_suppression, scale_coords, torch_utils


def _start_yolo(recvs, puts, cfg):
    with torch.no_grad():
        model_yolo, yolo_device = yolo_init(gpus=cfg.YOLO.GPU_ID,
                                            weights=cfg.YOLO.WEIGHTS,
                                            img_size=cfg.YOLO.IMG_SIZE,
                                            out=cfg.YOLO.OUTPUT,
                                            cfg=cfg.YOLO.CFG)
    while True:
        for recv_queue, put_queue in zip(recvs, puts):
            if not recv_queue.empty():
                imgs = recv_queue.get(timeout=0.2)
                res = detect(cfg.YOLO, imgs, model_yolo, yolo_device)
                put_queue.put(res)


def set_yolo(cfg, tasks=('attr',)):
    if not isinstance(tasks, tuple):
        raise ValueError('list all task with tuple.')

    recvs = []
    puts = []
    for t in tasks:
        assert t.lower() in ['attr', 'reid'], 'Unknown task: {:s}'.format(t)
        recv_queue = Queue(maxsize=1)
        put_queue = Queue(maxsize=1)
        recvs.append(recv_queue)
        puts.append(put_queue)
        if t.lower() == 'attr':
            recv_queue_1 = Queue(maxsize=1)
            put_queue_1 = Queue(maxsize=1)
            cfg.YOLO.FACE_RECV = recv_queue
            cfg.YOLO.FACE_PUT = put_queue
            cfg.YOLO.BODY_RECV = recv_queue_1
            cfg.YOLO.BODY_PUT = put_queue_1
            recvs.append(recv_queue_1)
            puts.append(put_queue_1)
        else:
            cfg.YOLO.REID_RECV = recv_queue
            cfg.YOLO.REID_PUT = put_queue

    pyolo = Process(target=_start_yolo, args=(recvs, puts, cfg))
    pyolo.start()


def yolo_init(gpus=0, weights='weights/yolov3.pt', img_size=1024, out='output', cfg='cfg/yolov3.cfg'):
    img_size = img_size  # (320, 192) or (416, 256) or (608, 352) for (height, width)

    # Initialize
    device = torch_utils.select_device(gpus=gpus, force_cpu=False)
    if os.path.exists(out):
        shutil.rmtree(out)  # delete output folder
    os.makedirs(out)  # make new output folder

    # Initialize model
    model = Darknet(cfg, img_size)

    # Load weights
    if weights.endswith('.pt'):  # pytorch format
        model.load_state_dict(torch.load(weights, map_location=device)['model'])
    else:  # darknet format
        _ = load_darknet_weights(model, weights)

    # Eval mode
    model.to(device).eval()

    return model, device


def detect(opt, img_list, model, device):
    img_size, out, source, weights, cfg, conf_thres, nms_thres, fourcc, data = opt.IMG_SIZE, opt.OUTPUT, opt.SOURCE, opt.WEIGHTS, opt.CFG, opt.CONF_TH, opt.NMS_TH, opt.FOURCC, opt.DATA_CFG
    person_results = []

    # Run inference
    for img_id, data in enumerate(img_list):
        cam_id = data[0]
        im0 = data[1]
        img = data[2]

        # Get detections
        img = torch.from_numpy(img).unsqueeze(0).to(device)
        pred, _ = model(img)
        det = non_max_suppression(pred.float(), conf_thres, nms_thres)[0]

        if det is not None and len(det):
            det[:, :4] = scale_coords(img.shape[2:], det[:, :4], im0.shape).round()
            for *xyxy, conf, _, cls in det:
                if cls == 0 and float('%.10f' % (conf)) >= 0.7:
                    x = xyxy
                    xmin = int(x[0])
                    ymin = int(x[1])
                    xmax = int(x[2])
                    ymax = int(x[3])
                    score = float('%.10f' % (conf))
                    person_results.append((cam_id, img_id, xmin, ymin, xmax, ymax, score))
    return person_results
