import queue
from multiprocessing import Process, Queue
from time import time

import cv2
import numpy as np

from utils import LOGGER
from yolov3.utils.datasets import letterbox


def process_img(img, img_size):
    img, *_ = letterbox(img, new_shape=img_size)
    img = img[:, :, ::-1].transpose(2, 0, 1)  # BGR to RGB
    img = np.ascontiguousarray(img, np.float32)  # uint8 to fp16/fp32
    img /= 255.0  # 0 - 255 to 0.0 - 1.0
    return img


def stream(isonline, url, q, img_size, fps):
    cam_id, uri = url
    first, rval, frame, cap = 1, False, None, None
    last = time()
    if isonline:
        while True:
            try:
                if rval:
                    # LOGGER.debug('read a frame in capture.')
                    if q.full():
                        # LOGGER.debug('queue is full while putting image, remove one image from queue.')
                        try:
                            q.get(timeout=0.5)
                        except queue.Empty:
                            LOGGER.debug('failed to remove image from queue.')
                    if time() - last > 1 / fps:
                        img = process_img(frame, img_size)
                        q.put([cam_id, frame, img])
                        last = time()
                    rval, frame = cap.read()
                else:
                    if cap is not None:
                        LOGGER.warning('reconnecting camera {:s}'.format(cam_id))

                    else:
                        LOGGER.info('initializing camera {:s}'.format(cam_id))
                    cap = cv2.VideoCapture(uri)
                    rval, frame = cap.read()

            except KeyboardInterrupt:
                # release camera then propagate KeyboardInterrupt signal
                if cap is not None:
                    cap.release()
                raise KeyboardInterrupt
    else:
        while True:
            try:
                if rval:
                    if not q.full():
                        img = process_img(frame, img_size)
                        q.put([cam_id, frame, img])
                        rval, frame = cap.read()
                        # frame = cv2.resize(frame, (1920, 1080))
                else:
                    if cap is not None:
                        LOGGER.warning('failed to open video file:{:s}'.format(cam_id))
                        assert cap is None, 'failed to open file or end of file!'
                        raise AssertionError
                    elif first:
                        LOGGER.info('opening file {:s}'.format(cam_id))
                        cap = cv2.VideoCapture(uri)
                        rval, frame = cap.read()
                        first=0
                        # frame = cv2.resize(frame, (1920, 1080))
                    else:
                        LOGGER.info('end of file:{:s}'.format(cam_id))
            except KeyboardInterrupt:
                if cap is not None:
                    cap.release()
                raise KeyboardInterrupt


def start_streams(cfg, tasks=('attr',)):
    if not isinstance(tasks, tuple):
        raise ValueError('list all task with tuple.')

    for t in tasks:
        assert t.lower() in ['attr', 'reid'], 'Unknown task: {:s}'.format(t)
        isOnline = (t.lower() is 'reid') and cfg.REID.ONLINE
        max_size_ = 32 if isOnline else 1
        urls = cfg.CAMERA.FACE_URLS if t.lower() == 'attr' else cfg.CAMERA.REID_URLS
        fps = cfg.ATTR.DET_FPS if t.lower() == 'attr' else cfg.REID.DET_FPS
        queues = []
        procs = []
        for url in urls:
            q = Queue(maxsize=max_size_)
            queues.append(q)
            procs.append(Process(target=stream, args=(isOnline, url, q, cfg.YOLO.IMG_SIZE, fps)))

        if t.lower() == 'attr':
            cfg.ATTR.IMG_QUEUE = queues
        else:
            cfg.REID.IMG_QUEUE = queues

        for proc in procs:
            proc.start()


def fetch(cfg, t='attr', completed=False):
    assert t.lower() in ['attr', 'reid'], 'Unknown task: {:s}'.format(t)
    imgs = []
    queue_list = getattr(cfg, t.upper()).IMG_QUEUE
    if completed:
        for i, q in enumerate(queue_list):
            if q.empty():
                LOGGER.debug('queue-{:d} is empty.'.format(i))
                return imgs
            else:
                LOGGER.debug('length of queue-{:} is: {:d}'.format(i, q.qsize()))

    for i, q in enumerate(queue_list):
        try:
            if not q.empty():
                imgs.append(q.get(timeout=0.5))
            else:
                LOGGER.debug('{:s} queue-{:d} is empty'.format(t, i))
        except queue.Empty:
            LOGGER.debug('{:s} wait queue-{:d} timeout'.format(t, i))
            continue
    return imgs


def debug_frame(paths, opt):
    cams = []
    caps = {}
    rvals = {}
    frames = {}
    for cam_id, path in paths:
        cams.append(cam_id)
        cap = cv2.VideoCapture(path)
        rval, frame = cap.read()
        caps[cam_id] = cap
        rvals[cam_id] = rval
        frames[cam_id] = frame

    while (all(rvals.values())):
        imgs = []
        for cam_id in cams:
            img, *_ = letterbox(frames[cam_id], new_shape=opt.YOLO.IMG_SIZE)
            img = img[:, :, ::-1].transpose(2, 0, 1)
            img = np.ascontiguousarray(img, dtype=np.float16 if opt.YOLO.HALF else np.float32)
            img /= 255.0
            imgs.append([cam_id, frames[cam_id], img])
            rval, frame = caps[cam_id].read()
            rvals[cam_id] = rval
            frames[cam_id] = frame

        yield imgs
